from flask import Flask, render_template, jsonify, request
import threading
import time
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler
from soco import discover
from rapidfuzz import fuzz
from typing import List

app = Flask(__name__)

class FileChangeHandler(FileSystemEventHandler):
    """Handles file change events for the banned songs and artists lists."""

    def __init__(self, controller):
        self.controller = controller

    def on_modified(self, event):
        if event.src_path.endswith(self.controller.banned_songs_file):
            print("Banned songs file changed. Reloading...")
            self.controller.banned_songs = self.controller._load_list_from_file(self.controller.banned_songs_file)
        elif event.src_path.endswith(self.controller.banned_artists_file):
            print("Banned artists file changed. Reloading...")
            self.controller.banned_artists = self.controller._load_list_from_file(self.controller.banned_artists_file)

class SonosController:
    def __init__(self, banned_songs_file: str, banned_artists_file: str, match_threshold: int = 85):
        self.banned_songs_file = banned_songs_file
        self.banned_artists_file = banned_artists_file
        self.match_threshold = match_threshold

        self.speakers = self._discover_speakers()
        self.banned_songs = self._load_list_from_file(self.banned_songs_file)
        self.banned_artists = self._load_list_from_file(self.banned_artists_file)

        # Monitoring state
        self.monitoring_enabled = True

        # Set up file monitoring
        self.file_observer = Observer()
        self.file_change_handler = FileChangeHandler(self)
        self.file_observer.schedule(self.file_change_handler, ".", recursive=False)
        self.file_observer.start()

    @staticmethod
    def _discover_speakers():
        """Discover Sonos speakers on the local network."""
        speakers = discover()
        if not speakers:
            raise RuntimeError("No Sonos speakers found on the network.")
        print(f"Discovered speakers: {[speaker.player_name for speaker in speakers]}")
        return speakers

    @staticmethod
    def _load_list_from_file(file_path: str) -> List[str]:
        """Load a list of strings (songs or artists) from a text file."""
        try:
            with open(file_path, "r") as f:
                return [line.strip().lower() for line in f if line.strip()]
        except FileNotFoundError:
            print(f"File '{file_path}' not found. No entries loaded.")
            return []

    def _is_banned(self, title: str, artist: str) -> bool:
        """Check if the title or artist matches any banned entry using fuzzy matching."""
        for banned in self.banned_songs:
            if fuzz.partial_ratio(title, banned) >= self.match_threshold:
                print(f"Match found: Title '{title}' matches banned song '{banned}'.")
                return True
        for banned in self.banned_artists:
            if fuzz.partial_ratio(artist, banned) >= self.match_threshold:
                print(f"Match found: Artist '{artist}' matches banned artist '{banned}'.")
                return True
        return False

    def get_current_track_info(self, speaker):
        """Get the current track's title and artist."""
        try:
            track_info = speaker.get_current_track_info()
            return track_info.get("title", "").lower(), track_info.get("artist", "").lower()
        except Exception:
            return "", ""

    def monitor_queue(self, check_interval=5):
        """Monitor the playback queue and handle banned songs or artists."""
        print("Monitoring Sonos playback...")
        while True:
            if not self.monitoring_enabled:
                time.sleep(1)
                continue
            for speaker in self.speakers:
                print(f"Checking speaker: {speaker.player_name}")
                title, artist = self.get_current_track_info(speaker)
                if self._is_banned(title, artist):
                    print(f"Banned track detected: '{title}' by '{artist}'. Stopping playback.")
                    try:
                        speaker.stop()
                    except Exception as e:
                        print(f"Error stopping playback: {e}")
            time.sleep(check_interval)

    def stop(self):
        """Stop the file observer."""
        self.file_observer.stop()
        self.file_observer.join()
        print("Stopped file observer.")

controller = None

@app.route("/")
def index():
    return render_template("index.html", 
                           monitoring_enabled=controller.monitoring_enabled,
                           banned_songs=controller.banned_songs,
                           banned_artists=controller.banned_artists)

@app.route("/toggle_monitoring", methods=["POST"])
def toggle_monitoring():
    controller.monitoring_enabled = not controller.monitoring_enabled
    return jsonify({"monitoring_enabled": controller.monitoring_enabled})

if __name__ == "__main__":
    # Specify the files containing the banned song and artist lists
    banned_songs_file = "banned_songs.txt"
    banned_artists_file = "banned_artists.txt"

    try:
        controller = SonosController(banned_songs_file, banned_artists_file)

        # Run the monitoring in a separate thread
        monitoring_thread = threading.Thread(target=controller.monitor_queue, daemon=True)
        monitoring_thread.start()

        # Start the Flask web server
        app.run(host="0.0.0.0", port=12345)
    finally:
        if controller:
            controller.stop()
